#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <sys/time.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_dflt_error.h"
#include "lf_scheduler.h"
#include "lf_xbar32.h"
#include "lf_product_def.h"
#include "libmyri.h"

#define LT_SEND_SIZE 1000
#define LT_PKT_TIMEOUT 100	/* inter-packet timeout in ms */
#define LT_PKT_TYPE 0x12345678
#define LT_MIN_PKTS_FOR_BW 100

/*
 * Private structs for linktest
 */

struct lt_xbar {
  int *bad;
  int *tested;
};
#define LT_XBAR(P) ((struct lt_xbar *)(P))

struct raw_pkt {
  uint32_t pkt_type;
  struct lf_tagged_xbar_insert tagged_xbar_stuff;
  unsigned int serial;
  unsigned int seqno;
};

enum {
  LT_RES_OK,
  LT_RES_NO_LINK,
  LT_RES_MARGINAL_LINK,
  LT_RES_SLOW_LINK
};

#define DELTA_TO_ROUTE(D) (((D)&0x3F) | 0x80)
#define ROUTE_TO_DELTA(R) (((unsigned char)(R)>0x90)?((R)-0xC0):((R)-0x80))

/*
 * prototypes
 */
static void check_all_links(struct lf_fabric *fp, struct lf_nic *np, int hport);
static int test_this_route(int ifc, unsigned char *route, int rlen, int is_tagged);
static int test_this_route(int ifc, unsigned char *route, int rlen, int report);
static int do_this_xbar(int ifc, struct lf_xbar *xp, int in_port, unsigned char *route,
	         int rlen, int depth);

static char *One_switch;
static int Num_packets;
static double BWthresh;
static int Show_route;
static int Hport;
static int Myri;
static int Ntests;
static int Nic_id;

void
parse_args(
  int argc,
  char **argv)
{
  int c;
  extern char *optarg;

  Hport = -1;			/* -1 means all */
  Num_packets = 4;
  BWthresh = 20.;
  One_switch = NULL;
  Nic_id = 0;

  while ((c = getopt(argc, argv, "S:ri:l:t:n:")) != EOF) switch (c) {
  case 'S':
    One_switch = optarg;
    break;

  case 'r':
    Show_route = 1;
    break;

  case 'i':  /* Host port Id to test out of */
    Hport = atoi(optarg);
    break;

  case 'n':
    Nic_id = atoi(optarg);
    break;

  case 'l':
    Num_packets = atoi(optarg);
    break;

  case 't':
    BWthresh = atoi(optarg);
    break;

  }

  printf("Testing each link with %d packets\n", Num_packets);

  if (Num_packets < LT_MIN_PKTS_FOR_BW) {
    printf("Notice: Since number of packets (%d) is less than %d,\n",
           Num_packets, LT_MIN_PKTS_FOR_BW);
    printf("        BW check will not be performed.\n");
  }
}

/*
 * Find this host in the fabric list of hosts.  We do this by
 * finding the NIC matching our NIC's MAC address, then go to its parent
 * to get the host struct
 */
static struct lf_host *
find_this_host(
  struct lf_fabric *fp)
{
  struct myri_nic_info info;
  struct lf_nic *np;
  struct lf_host *hp;
  int rc;

  /* get information including MAC address about this NIC */
  rc = myri_get_nic_info(Myri, &info);
  if (rc != 0) LF_ERROR(("Getting NIC info"));

  /* find matching NIC in fabric */
  np = lf_find_nic_by_mac(fp, info.mac_addr);
  if (np == NULL) LF_ERROR(("Cannot find my NIC in fabric"));

  /* get parent for this NIC */
  hp = np->host;

  return hp;

 except:
  exit(1);
}


int
main(
  int argc,
  char **argv)
{
  struct lf_fabric *fp;
  struct lf_host *myhp;
  struct lf_nic *nicp;
  int ifc;

  lf_init();
  parse_args(argc, argv);

  Myri = myri_open(Nic_id);
  if (Myri == -1) {
    perror("myri_open");
    exit(1);
  }


  fp = lf_simple_load_fabric();
  if (fp == NULL) {
    fprintf(stderr, "Error loading fabric\n");
    exit(1);
  }

  myhp = find_this_host(fp);
  nicp = lf_find_nic_by_id(myhp, Nic_id);
  if (nicp == NULL) {
    fprintf(stderr, "No NIC with ID %d in this host\n", Nic_id);
    exit(1);
  }

  /* If user asked to test out of a single port instead of all ports,
   * make sure that port exists according to the map.
   */
  if (Hport != -1) {
    if (Hport >= nicp->num_ports) {
      fprintf(stderr, "Requested host interface number too big\n");
      exit(1);
    }
  }

  if (Hport == -1) {
    for (ifc = 0; ifc < nicp->num_ports; ifc++) {
      check_all_links(fp, nicp, ifc);
    }
  } else {
    /* Just do checking out of whatever port Hport is set to */
    check_all_links(fp, nicp, Hport);
  }

  /* myri_close(Myri); why disabled?  i forget... */

  return 0;
}   


static void
print_route(
  unsigned char *route,
  int rlen)
{
  int i;

  printf("route:");
  for (i=0; i<rlen; ++i) {
    printf(" %d", LF_ROUTE_TO_DELTA(route[i]));
  }
  printf("\n");
}

static void
print_result(
  struct lf_xbar *xp,
  int port,
  int result)
{
  struct lf_xbar *oxp;
  int oport;
  struct lf_linecard *lp;
  struct lf_linecard *olp;

  oxp = LF_XBAR(xp->topo_ports[port]);
  oport = xp->topo_rports[port];

  lp = xp->linecard;
  olp = oxp->linecard;


  /* If phys connection is same xbar, then it's an internal link */
  if (xp->phys_ports[port] == LF_NODE(oxp)) {

    printf("The internal link on enclosure %s between\n",
	lp->enclosure->name);
    printf("\tslot %d, xbar %d, port %d    and\n",
	lf_slot_display_no(lp), xp->xbar_no, port);
    printf("\tslot %d, xbar %d, port %d\n",
	lf_slot_display_no(olp), oxp->xbar_no, oport);

  } else {
    struct lf_xcvr *xcp;
    struct lf_xcvr *oxcp;
    int xcport;

    xcp = LF_XCVR(xp->phys_ports[port]);
    oxcp = LF_XCVR(oxp->phys_ports[oport]);

    /* convert internal port number to external */
    xcport = xp->phys_rports[port] - xcp->num_conns;

    /* sanity checks */
    if (xcp->ln_type != LF_NODE_LC_XCVR) {
      printf("%s,s%d,x%d,p%d not connected to xcvr??\n", lp->enclosure->name,
	lf_slot_display_no(lp), xp->xbar_no, port);
      exit(1);
    }
    if (oxcp->ln_type != LF_NODE_LC_XCVR) {
      printf("%s,s%d,x%d,p%d not connected to xcvr??\n", olp->enclosure->name,
	lf_slot_display_no(olp), oxp->xbar_no, oport);
      exit(1);
    }
    if (xcp->ports[xcport] != LF_NODE(oxcp)) {
      printf("xcvrs not connected as expected??\n");
      printf("btwn %s,s%d,x%d,p%d", lp->enclosure->name,
	lf_slot_display_no(lp), xp->xbar_no, port);
      printf("and %s,s%d,x%d,p%d\n", olp->enclosure->name,
	lf_slot_display_no(olp), oxp->xbar_no, oport);
      exit(1);
    }

    /* change the linecard pointers to be for the xcvrs */
    lp = xcp->p.linecard;
    olp = oxcp->p.linecard;

    printf("The %s link between\n", lp->def->link_type);
    printf("\t%s, slot %d, port %d:%d   and\n",
	lp->enclosure->name, lf_slot_display_no(lp),
	lp->xcvr_labels[xcp->port],
	xp->phys_rports[port] - xcp->num_conns);
    printf("\t%s, slot %d, port %d:%d\n",
	olp->enclosure->name, lf_slot_display_no(olp),
	olp->xcvr_labels[oxcp->port],
	oxp->phys_rports[oport] - oxcp->num_conns);

  }

  /* Now say what's wrong with the link */
  if (result == LT_RES_NO_LINK) {
    printf("is currently passing no traffic.\n");
  } else if (result == LT_RES_MARGINAL_LINK) {
    printf("is exhibiting high packet loss\n");
  } else if (result == LT_RES_SLOW_LINK) {
    printf("is passing traffic slowly.\n");
  }

  printf("\n");
}

static void inline
alloc_xbar_user(
  struct lf_xbar *xp)
{
  struct lt_xbar *up;

  if (xp->user.v != NULL) return;

  LF_CALLOC(up, struct lt_xbar, 1);
  LF_CALLOC(up->bad, int, xp->num_ports);
  LF_CALLOC(up->tested, int, xp->num_ports);

  xp->user.v = up;
  return;

 except:
  exit(1);
}

static int
do_this_xbar(
  int ifc,
  struct lf_xbar *xp,
  int in_port,
  unsigned char *route,
  int rlen,
  int depth)
{
  int i;
  struct lf_xbar *newxp;
  int new_port;
  int count;
  int result;
  int is_tagged;

  /* printf("doing xbar %s, in_port: %d, depth: %d\n", xp->name, in_port, depth); */

  alloc_xbar_user(xp);		/* make sure user struct exists for this xbar */

  /* if depth is zero, this is a leaf, execute the tests */
  if (depth == 0) {

    if (One_switch == NULL
	|| strcmp(xp->linecard->enclosure->name, One_switch) == 0) {


      ++Ntests;	/* keep track of links tested */

      is_tagged = (xp->xbar_id != 0);

      /* perform the test, report if fails */
      result = test_this_route(ifc, route, rlen, is_tagged);

      if (result != LT_RES_OK) {

	/* mark both ends bad */
	LT_XBAR(xp->user.v)->bad[in_port] = TRUE;
	if (xp->topo_ports[in_port]->ln_type == LF_NODE_XBAR) {
	  struct lf_xbar *oxp;

	  oxp = LF_XBAR(xp->topo_ports[in_port]);
	  alloc_xbar_user(oxp);
	  LT_XBAR(oxp->user.v)->bad[xp->topo_rports[in_port]] = TRUE;
	}

	if (Show_route) print_route(route, rlen);
	print_result(xp, in_port, result);
      }
    }

    return 1;



  /* non-zero depth means to branch out to next level of non-tested xbars */
  } else {
    int end_port;

    count = 0;		/* count how many new ports we tested */

    /* if we came in on a port > 15, then this is an xbar 32 and
     * we should not go out any ports > 15
     */
    if (in_port > 15) {
      end_port = 16;
    } else {
      end_port = xp->num_ports;
    }

    /* Now, work on each attached xbar */
    for (i=0; i<end_port; ++i) {

      /* skip non-xbars and bad ports */
      if (xp->topo_ports[i] == NULL
	  || xp->topo_ports[i]->ln_type != LF_NODE_XBAR
	  || LT_XBAR(xp->user.v)->bad[i]) {
	continue;
      }

      /* xbar and port we will process next */
      newxp = LF_XBAR(xp->topo_ports[i]);
      new_port = xp->topo_rports[i];

      /* route to the new xbar */
      route[rlen] = DELTA_TO_ROUTE(i - in_port);

      /* depth of 1 means the links will be tested */
      if (depth == 1) {

	/* If this port already tested, continue */
	if (LT_XBAR(xp->user.v)->tested[i]) {
	  continue;
	}
	alloc_xbar_user(newxp);

	LT_XBAR(xp->user.v)->tested[i] = TRUE;	/* this link has been tested */
	LT_XBAR(newxp->user.v)->tested[new_port] = TRUE;
      }

      /* process the xbar */
      count += do_this_xbar(ifc, newxp, new_port, route, rlen+1, depth-1);
    }
  }

  /* return the number of tests performed */
  return count;
}

void
check_all_links(
  struct lf_fabric *fp,
  struct lf_nic *nicp,
  int ifc)
{
  struct lf_xbar *xb;
  unsigned char route[32];
  int port;
  int depth;
  int result;
  int rc;

  /* This is the host we start from */
  xb = LF_XBAR(nicp->topo_ports[ifc]);
  port = nicp->topo_rports[ifc];

  /* make sure we have connection - OK to say not tagged */
  result = test_this_route(ifc, route, 0, 0);
  if (result != LT_RES_OK) {
    printf("This host seems disconnected.\n");
    return;
  }

  /* keep running the checker at increasing depths until nothing found */
  Ntests = 0;
  depth = 1;
  do {
    rc = do_this_xbar(ifc, xb, port, route, 0, depth);
    ++depth;
  } while (rc > 0);

  printf ("%d links checked on interface %d.\n", Ntests, ifc);
}



char pop[]="|/-\\";
int popi;

int
test_this_route(
  int ifc,
  unsigned char *route,
  int rlen,
  int is_tagged)
{
  int i;
  int r;
  static unsigned int serial_no = 0;
  struct raw_pkt pkt;
  int pass;
  double bw;
  int elapsed_ms;
  struct timeval start;
  struct timeval end;
  int rx_packets;
  int rc;

  /* spin a little wheel */
  putchar(pop[popi++&3]);
  putchar(8);
  fflush(stdout);

  /* default to OK */
  pass = LT_RES_OK;

  ++serial_no;		/* increment serial number for this test */

  /* fill out the route with a U-turn byte and return route */
  route[rlen] = is_tagged ? 0xa0 : 0x80;
  for  (r=0; r<rlen; ++r) {
    route[rlen+r+1] = LF_DELTA_TO_ROUTE(-LF_ROUTE_TO_DELTA(route[rlen-r-1]));
  }
  rlen = rlen*2 + 1;		/* update rlen */

  /* zero out the bytes to be replaced by the tagged xbar HW so the
   * CRC32 comes out right (do always, even though un-needed is !is_tagged) */
  memset(&pkt.tagged_xbar_stuff, 0, sizeof(pkt.tagged_xbar_stuff));

  /* We will send Num_packets 1000 byte packets and then keep receiving
   * packets until either:
   *   a) last send packet is received OR
   *   b) PKT_TIMEOUT milliseconds have elapsed since last received packet
   */
  rx_packets = 0;
  
  /* launch all sends */
  pkt.pkt_type = LT_PKT_TYPE;
  pkt.serial = serial_no;
  for (i=0; i<Num_packets; ++i) {
    pkt.seqno = i;
    rc = myri_raw_send(Myri, ifc, route, rlen, &pkt, LT_SEND_SIZE, NULL);
    if (rc == -1) {
      perror("myri_raw_send");
      exit(1);
    }
  }

  /* record when we start looking for packets */
  rc = gettimeofday(&start, NULL);
  if (rc == -1) {
    perror("gettimeofday");
    exit(1);
  }

  /* consume packets until all received or until myri_next_event() times out */
  while (1) {
    struct myri_event *mep;

    /* get next event with a timeout */
    rc = myri_next_event(Myri, &mep, LT_PKT_TIMEOUT);
    if (rc == -1) {
      perror("myri_next_event");
      exit(1);
    }

    /* no event means timeout, so we're done */
    if (mep->type == MYRI_EVENT_NO_EVENT) {
      /* no need to release non-events */
      break;
    }

    /* process any receives */
    if (mep->type == MYRI_EVENT_RAW_RECV_COMPLETE) {
      struct raw_pkt *pp;
      pp = mep->d.raw_recv.rxbuf;

      /* process this packet if it's ours */
      if (pp->pkt_type == LT_PKT_TYPE && pp->serial == serial_no) { 
	++rx_packets;
	if (pp->seqno == Num_packets - 1) {
	  myri_release_event(mep);		/* release this event */
	  break;
	}
      }
    }

    myri_release_event(mep);		/* release this event */
  }


  /* done, for better or worse, mark end time */
  rc = gettimeofday(&end, NULL);
  if (rc == -1) {
    perror("gettimeofday");
    exit(1);
  }

  /* compute bandwidth (MB/s) */
  elapsed_ms = tv_diff(&end, &start);
  if (elapsed_ms > 0) {
    bw = (LT_SEND_SIZE * Num_packets) / tv_diff(&end, &start) / 1000;
  } else {
    bw = 1000;
  }

  if (rx_packets == 0) {
    pass = LT_RES_NO_LINK;

  } else if (Num_packets - rx_packets > 0) {
    printf("Dropped %d / %d packets.\n", Num_packets - rx_packets, Num_packets);
    pass = LT_RES_MARGINAL_LINK;

  } else if (Num_packets >= LT_MIN_PKTS_FOR_BW && bw < BWthresh) {
    printf("BW = %f MB/s\n", bw);

    pass = LT_RES_SLOW_LINK;
  }

  return pass;
}
